/*******************************************************************************
* Copyright (c) 2020 Mingyao Chuang
*
* All rights reserved.
* This program is free to use, but the ban on selling behavior.
* Modify the program must keep all the original text description,
* and can only comment out the original code (not allowed to delete).
*
* e-mail:mingyaochuang@gmail.com
*******************************************************************************/
'use strict';

goog.provide('Blockly.Blocks.dbsp');

goog.require('Blockly.Blocks');

Blockly.Blocks.dbsp.HUE = 300;
Blockly.Blocks.dbsp.INPUT_HUE = 120;
Blockly.Blocks.dbsp.INSTANCE_NAME = 'dbsp_';
Blockly.Blocks.dbsp.GLOBAL = {
	baud_select: [["57600", "57600"], ["115200", "115200"]],
	servo_select: [["1", "1"], ["2", "2"], ["3", "3"], ["4", "4"], ["5", "5"], ["6", "6"]]
};

Blockly.Blocks.dbsp.vars_ = Object.create(null);

Blockly.Blocks.dbsp.initCheck = function (obj) {
	var count = Blockly.Blocks.dbsp.getInitObjectCount();
	if (count === 0) {
		obj.setWarningText(Blockly.DBSP_NEED_INIT);
		return false;
	}
	else if (count === 1) {
		return true;
	}
	else {
		obj.setWarningText(Blockly.DBSP_NOT_MULITI_INIT);
		return false;
	}
};

Blockly.Blocks.dbsp.updateCheck = function (obj) {
	var count = Blockly.Blocks.dbsp.getUpdateObjectCount();
	if (count === 0) {
		obj.setWarningText(Blockly.DBSP_NEED_UPDATE_BLOCK);
		return false;
	}
	else if (count === 1) {
		return true;
	}
	else {
		obj.setWarningText(Blockly.DBSP_NOT_MULITI_UPDATE_BLOCK);
		return false;
	}
};

Blockly.Blocks.dbsp.checkInitBlocks = function (element) {
	if (element.tag === 'dbsp_begin') {
		++Blockly.Blocks.dbsp.vars_['initObjectCount'];
	}
};

Blockly.Blocks.dbsp.checkUpdateBlocks = function (element) {
	if (element.tag === 'dbsp_update') {
		++Blockly.Blocks.dbsp.vars_['updateObjectCount'];
	}
};

Blockly.Blocks.dbsp.getInitObjectCount = function () {
	Blockly.Blocks.dbsp.vars_['initObjectCount'] = 0;
	Blockly.mainWorkspace.getAllBlocks().forEach(Blockly.Blocks.dbsp.checkInitBlocks);
	return Blockly.Blocks.dbsp.vars_['initObjectCount'];
};

Blockly.Blocks.dbsp.getUpdateObjectCount = function () {
	Blockly.Blocks.dbsp.vars_['updateObjectCount'] = 0;
	Blockly.mainWorkspace.getAllBlocks().forEach(Blockly.Blocks.dbsp.checkUpdateBlocks);
	return Blockly.Blocks.dbsp.vars_['updateObjectCount'];
};

Blockly.Blocks.dbsp.setWarningText = function (element, text) {
	if (text === null) {
		return true;
	}
	else {
		element.setWarningText(text);
		return false;
	}
};

Blockly.Blocks.dbsp.servoIdCheck = function (element, fieldName, defaultValueText, includeAll = false) {
	var target = element.getInput(fieldName).connection.targetBlock();
	if (target !== null) {
		if (target.tag === 'dbsp_servo_id_inputer') {
			return Blockly.Blocks.dbsp.setWarningText(element, null);
		}
		else {
			if (target.type === "variables_get") {
				//return Blockly.Blocks.dbsp.setWarningText(element, Blockly.DBSP_MAKE_SURE_VAR);
				return Blockly.Blocks.dbsp.setWarningText(element, null);
			}
			else {
				var id = parseInt(Blockly.Arduino.valueToCode(element, fieldName));
				var stream = id >> 4;
				var order = id & 0xf;
				var includeAllCase = includeAll ? id === 0xff : false;
				if (includeAllCase || (stream >= 1 && stream <= 6 && order >= 1 && order <= 6)) {
					return Blockly.Blocks.dbsp.setWarningText(element, null);
				}
				else {
					return Blockly.Blocks.dbsp.setWarningText(element, Blockly.DBSP_VALUE_NOT_SUPPORTED + '(' + Blockly.DBSP_SERVO_ID + ')');
				}
			}
		}
	}
	else {
		return Blockly.Blocks.dbsp.setWarningText(element, Blockly.DBSP_NEED_INPUT_BLOCK + defaultValueText);
	}
};

Blockly.Blocks.dbsp.numberCheck = function (element, fieldName, fieldText, max, min) {
	var target = element.getInput(fieldName).connection.targetBlock();
	if (target !== null) {
		if (target.tag === 'dbsp_servo_id_inputer') {
			return Blockly.Blocks.dbsp.setWarningText(element, null);
		}
		else {
			if (target.type === "variables_get") {
				return Blockly.Blocks.dbsp.setWarningText(element, null);
			}
			else {
				var val = parseInt(Blockly.Arduino.valueToCode(element, fieldName));
				if (val >= min && val <= max) {
					return Blockly.Blocks.dbsp.setWarningText(element, null);
				}
				else {
					return Blockly.Blocks.dbsp.setWarningText(element, val + ' ' + Blockly.DBSP_VALUE_NOT_SUPPORTED + '(' + fieldText + ')');
				}
			}
		}
	}
	else {
		return Blockly.Blocks.dbsp.setWarningText(element, Blockly.DBSP_NEED_INPUT_BLOCK);
	}
};

Blockly.Blocks.dbsp_servo_id_all = {
	init: function () {
		this.tag = 'dbsp_servo_id_all';
		this.setColour(Blockly.Blocks.dbsp.INPUT_HUE);
		this.appendDummyInput()
			.appendField(Blockly.DBSP_ALL_SERVO);
		this.setOutput(true, Number);
		this.setTooltip(Blockly.DBSP_TOOLTIP_SERVO_ID_INPUTER);
	},

	onchange: function (changeEvent) {
		var parent = this.getParent();
		if (parent !== null) {
			this.setWarningText(null);
		}
		else {
			this.setWarningText(Blockly.DBSP_NOT_EXIST_ALONE);
		}
	}
};

Blockly.Blocks.dbsp_servo_id_inputer = {
	init: function () {
		this.tag = 'dbsp_servo_id_inputer';
		this.setColour(Blockly.Blocks.dbsp.INPUT_HUE);
		this.appendDummyInput('sid')
			.appendField("#")
			.appendField(new Blockly.FieldDropdown(Blockly.Blocks.dbsp.GLOBAL.servo_select), 'stream')
			.appendField("-")
			.appendField(new Blockly.FieldDropdown(Blockly.Blocks.dbsp.GLOBAL.servo_select), 'order');
		this.setOutput(true, Number);
		this.setTooltip(Blockly.DBSP_TOOLTIP_SERVO_ID_INPUTER);
	},

	onchange: function (changeEvent) {
		var parent = this.getParent();
		if (parent !== null) {
			this.setWarningText(null);
		}
		else {
			this.setWarningText(Blockly.DBSP_NOT_EXIST_ALONE);
		}
	}
};

Blockly.Blocks.dbsp_action_content_inputer = {
	init: function () {
		this.tag = 'dbsp_action_content_inputer';
		this.setColour(Blockly.Blocks.dbsp.INPUT_HUE);
		this.appendDummyInput()
			.appendField(Blockly.DBSP_SERVO_ROTATE)
			.appendField("#")
			.appendField(new Blockly.FieldDropdown(Blockly.Blocks.dbsp.GLOBAL.servo_select), 'stream')
			.appendField("-")
			.appendField(new Blockly.FieldDropdown(Blockly.Blocks.dbsp.GLOBAL.servo_select), 'order')
			.appendField(Blockly.DBSP_ANGLE)
			.appendField(new Blockly.FieldNumber(0, -180, 180), 'degree')
			.appendField(Blockly.DBSP_ANGLE_UNIT)
			.appendField(Blockly.DBSP_INTERVAL)
			.appendField(new Blockly.FieldNumber(1000, 0, 100000), 'interval')
			.appendField(Blockly.DBSP_MILLISECOND);
		this.setPreviousStatement(true, null);
		this.setNextStatement(true, null);
		this.setTooltip(Blockly.DBSP_TOOLTIP_ACTION_CONTENT_INPUTER);
	},

	onchange: function (changeEvent) {
		var root = this.getRootBlock();
		if (root !== null && root.tag === 'dbsp_action') {
			this.setWarningText(null);
		}
		else {
			this.setWarningText(Blockly.DBSP_NOT_EXIST_ALONE);
		}
	}
};

Blockly.Blocks.dbsp_begin = {
	init: function () {
		this.tag = 'dbsp_begin';
		this.setColour(Blockly.Blocks.dbsp.HUE);
		this.appendDummyInput()
			.appendField(Blockly.DBSP_PRODUCT_NAME + ' ' + Blockly.DBSP_INIT);
			//.appendField(new Blockly.FieldImage("../../media/dbsp/dbsp_board.png", 40, 32));
		this.appendDummyInput()
			.appendField(Blockly.DBSP_SERIAL_PORT)
			.appendField(new Blockly.FieldDropdown(profile.default.serial_select, this.handleSelect.bind(this)), 'serial')
			.appendField(Blockly.DBSP_BAUD)
			.appendField(new Blockly.FieldDropdown(Blockly.Blocks.dbsp.GLOBAL.baud_select), 'baud');
		this.appendDummyInput()
			.appendField(Blockly.DBSP_DELAY_AFTER_INIT)
			.appendField(new Blockly.FieldNumber(3000, 0, 100000), 'delay')
			.appendField(Blockly.DBSP_MILLISECOND);
		this.appendDummyInput('pins')
			.appendField("RX#")
			.appendField(Blockly.MIXLY_PIN)
			.appendField(new Blockly.FieldDropdown(profile.default.digital), 'rx')
			.appendField("TX#")
			.appendField(Blockly.MIXLY_PIN)
			.appendField(new Blockly.FieldDropdown(profile.default.digital), 'tx');
		this.serialName = this.getFieldValue('serial');
		this.isSoftwareSerial = !this.serialName.startsWith('Se');
		this.updateBlock();
		this.setPreviousStatement(true, null);
		this.setNextStatement(true, null);
		this.setTooltip(Blockly.DBSP_TOOLTIP_BEGIN);
	},

	onchange: function (changeEvent) {
		if (Blockly.Blocks.dbsp.getInitObjectCount() <= 1) {
			this.setWarningText(null);
		}
		else {
			this.setWarningText(Blockly.DBSP_NOT_MULITI_INIT);
		}
	},

	handleSelect: function (newSerial) {
		if (this.serialName !== newSerial) {
			this.serialName = newSerial;
			this.isSoftwareSerial = !this.serialName.startsWith('Se');
			if (this.isSoftwareSerial && newSerial !== "mySerial") {
				return null;
			}
			this.updateBlock();
		}
	},

	updateBlock: function () {
		this.getInput('pins').setVisible(this.isSoftwareSerial);
	},

	mutationToDom: function () {
		var container = document.createElement('mutation');
		container.setAttribute('serialname', this.serialName);
		container.setAttribute('issoftwareserial', this.isSoftwareSerial);
		return container;
	},

	domToMutation: function (xmlElement) {
		var serialName = xmlElement.getAttribute('serialname');
		if (serialName && serialName !== 'undefined') {
			this.serialName = serialName;
			this.isSoftwareSerial = !this.serialName.startsWith('Se');
		}
		this.updateBlock();
	}
};

Blockly.Blocks.dbsp_update = {
	init: function () {
		this.tag = 'dbsp_update';
		this.setColour(Blockly.Blocks.dbsp.HUE);
		this.appendDummyInput()
			.appendField(Blockly.DBSP_UPDATE);
		this.setPreviousStatement(true);
		this.setNextStatement(true);
		this.setTooltip(Blockly.DBSP_TOOLTIP_DBSP_UPDATE);
	},

	onchange: function (changeEvent) {
		if (Blockly.Blocks.dbsp.initCheck(this)) {
			if (Blockly.Blocks.dbsp.getUpdateObjectCount() <= 1) {
				this.setWarningText(null);
			}
			else {
				this.setWarningText(Blockly.DBSP_NOT_MULITI_UPDATE_BLOCK);
			}
		}
	}
};

Blockly.Blocks.dbsp_readAngle = {
	init: function () {
		this.tag = 'dbsp_readAngle';
		this.setColour(Blockly.Blocks.dbsp.HUE);
		this.appendDummyInput()
			.appendField(Blockly.DBSP_READ_ANGLE);
		this.appendValueInput('sid', Number)
			.setCheck(Number);
		this.setInputsInline(true);
		this.setOutput(true, Number);
		this.setTooltip(Blockly.DBSP_TOOLTIP_READ_ANGLE);
	},

	onchange: function (changeEvent) {
		if (Blockly.Blocks.dbsp.initCheck(this)) {
			if (this.getParent() !== null) {
				if (Blockly.Blocks.dbsp.servoIdCheck(this, 'sid', Blockly.DBSP_SERVO_ID_DEFAULT2)) {
					this.setWarningText(null);
				}
			}
			else {
				this.setWarningText(Blockly.DBSP_ASSIGN_VAR);
			}
		}
	}
};

Blockly.Blocks.dbsp_rotate = {
	init: function () {
		this.tag = 'dbsp_rotate';
		this.setColour(Blockly.Blocks.dbsp.HUE);
		this.appendValueInput('sid', Number)
			.appendField(Blockly.DBSP_SERVO)
			.setCheck(Number);
		this.appendValueInput('degree', Number)
			.appendField(Blockly.DBSP_ANGLE)
			.setCheck(Number);
		this.appendDummyInput()
			.appendField(Blockly.DBSP_ANGLE_UNIT);
		this.appendValueInput('interval', Number)
			.appendField(Blockly.DBSP_INTERVAL)
			.setCheck(Number);
		this.appendDummyInput()
			.appendField(Blockly.DBSP_MILLISECOND);
		this.setPreviousStatement(true, null);
		this.setNextStatement(true, null);
		this.setTooltip(Blockly.DBSP_TOOLTIP_ROTATE);
	},

	onchange: function (changeEvent) {
		if (Blockly.Blocks.dbsp.initCheck(this) &&
			Blockly.Blocks.dbsp.servoIdCheck(this, 'sid', Blockly.DBSP_SERVO_ID_DEFAULT2) &&
			Blockly.Blocks.dbsp.numberCheck(this, 'degree', Blockly.DBSP_ANGLE, 180, -180) &&
			Blockly.Blocks.dbsp.numberCheck(this, 'interval', Blockly.DBSP_INTERVAL, 100000, 0)) {
			this.setWarningText(null);
		}
	}
};

Blockly.Blocks.dbsp_rotate_by_speed = {
	init: function () {
		this.tag = 'dbsp_rotate_by_speed';
		this.setColour(Blockly.Blocks.dbsp.HUE);
		this.appendValueInput('sid', Number)
			.appendField(Blockly.DBSP_SERVO)
			.setCheck(Number);
		this.appendValueInput('degree', Number)
			.appendField(Blockly.DBSP_ANGLE)
			.setCheck(Number);
		this.appendDummyInput()
			.appendField(Blockly.DBSP_ANGLE_UNIT);
		this.appendValueInput('speed', Number)
			.appendField(Blockly.DBSP_SPEED)
			.setCheck(Number);
		this.appendDummyInput()
			.appendField(Blockly.DBSP_PER_SECOND_DEGREE);
		this.setPreviousStatement(true, null);
		this.setNextStatement(true, null);
		this.setTooltip(Blockly.DBSP_TOOLTIP_ROTATE_BY_SPEED);
	},

	onchange: function (changeEvent) {
		if (Blockly.Blocks.dbsp.initCheck(this) && 
			Blockly.Blocks.dbsp.updateCheck(this) &&
			Blockly.Blocks.dbsp.servoIdCheck(this, 'sid', Blockly.DBSP_SERVO_ID_DEFAULT2) &&
			Blockly.Blocks.dbsp.numberCheck(this, 'degree', Blockly.DBSP_ANGLE, 180, -180) &&
			Blockly.Blocks.dbsp.numberCheck(this, 'speed', Blockly.DBSP_INTERVAL, 1000, 0)) {
			this.setWarningText(null);
		}
	}
};

Blockly.Blocks.dbsp_action = {
	init: function () {
		this.tag = 'dbsp_action';
		this.actionName = '';
		this.setColour(Blockly.Blocks.dbsp.HUE);
		this.appendDummyInput()
			.appendField(Blockly.DBSP_ACTION)
			.appendField(new Blockly.FieldTextInput('action'), 'aname');
		this.appendStatementInput('content')
			.setCheck('String');
		this.setTooltip(Blockly.DBSP_TOOLTIP_ACTION);
	},

	onchange: function (changeEvent) {
		this.actionName = this.getFieldValue('aname');
		if (Blockly.Blocks.dbsp.initCheck(this)) {
			if (this.getInput('content').connection.targetBlock()) {
				if (this.getActionTitleCount() > 1) {
					this.setWarningText(Blockly.DBSP_THE_SAME_ACTION_NAME);
				}
				else {
					this.setWarningText(null);
				}
			}
			else {
				this.setWarningText(Blockly.DBSP_NEED_ACTION_CONTENT);
			}
		}
	},

	checkTitle: function (element) {
		if (element.tag === 'dbsp_action') {
			++Blockly.Blocks.dbsp.vars_['actionTitleCount' + element.actionName];
		}
	},

	getActionTitleCount: function () {
		Blockly.Blocks.dbsp.vars_['actionTitleCount' + this.actionName] = 0;
		Blockly.mainWorkspace.getAllBlocks().forEach(this.checkTitle);
		return Blockly.Blocks.dbsp.vars_['actionTitleCount' + this.actionName];
	}	
};

Blockly.Blocks.dbsp_execute_action = {
	init: function () {
		this.tag = 'dbsp_execute_action';
		this.setColour(Blockly.Blocks.dbsp.HUE);
		this.appendDummyInput()
			.appendField(Blockly.DBSP_EXECUTE_ACTION)
			.appendField(Blockly.DBSP_NUMBER)
			.appendField(new Blockly.FieldNumber(0, 0, 9999999999), 'aid');
		this.setPreviousStatement(true, null);
		this.setNextStatement(true, null);
		this.setTooltip(Blockly.DBSP_TOOLTIP_EXECUTE_ACTION);
	},

	onchange: function (changeEvent) {
		if (Blockly.Blocks.dbsp.initCheck(this)) {
			this.setWarningText(null);
		}
	}
};

Blockly.Blocks.dbsp_execute_action_array = {
	init: function () {
		this.tag = 'dbsp_execute_action_array';
		this.setColour(Blockly.Blocks.dbsp.HUE);
		this.appendDummyInput()
			.appendField(Blockly.DBSP_EXECUTE_ACTION)
			.appendField(Blockly.DBSP_ACTION_NAME)
			.appendField(new Blockly.FieldTextInput('action'), 'aname');
		this.setPreviousStatement(true, null);
		this.setNextStatement(true, null);
		this.setTooltip(Blockly.DBSP_TOOLTIP_EXECUTE_ACTION_ARRAY);
	},

	onchange: function (changeEvent) {
		if (Blockly.Blocks.dbsp.initCheck(this)) {
			var name = this.getFieldValue('aname');
			var servoCount = Blockly.Blocks.dbsp.vars_['dbsp_action_servo_count' + name] || 0;
			if (servoCount > 0) {
				this.setWarningText(null);
			}
			else {
				this.setWarningText(Blockly.DBSP_INVALID_ACTION_NAME);
			}
		}
	}
};

Blockly.Blocks.dbsp_execute_macro = {
	init: function () {
		this.tag = 'dbsp_execute_macro';
		this.setColour(Blockly.Blocks.dbsp.HUE);
		this.appendDummyInput()
			.appendField(Blockly.DBSP_EXECUTE_MACRO)
			.appendField(Blockly.DBSP_NUMBER)
			.appendField(new Blockly.FieldNumber(0, 0, 9999999999), 'mid');
		this.setPreviousStatement(true, null);
		this.setNextStatement(true, null);
		this.setTooltip(Blockly.DBSP_TOOLTIP_EXECUTE_MACRO);
	},

	onchange: function (changeEvent) {
		if (Blockly.Blocks.dbsp.initCheck(this)) {
			this.setWarningText(null);
		}
	}
};

Blockly.Blocks.dbsp_release = {
	init: function () {
		this.tag = 'dbsp_release';
		this.setColour(Blockly.Blocks.dbsp.HUE);
		this.appendValueInput('sid', Number)
			.appendField(Blockly.DBSP_RELEASE)
			.setCheck(Number);
		this.setPreviousStatement(true, null);
		this.setNextStatement(true, null);
		this.setTooltip(Blockly.DBSP_TOOLTIP_RELEASE);
	},

	onchange: function (changeEvent) {
		if (Blockly.Blocks.dbsp.initCheck(this) &&
			Blockly.Blocks.dbsp.servoIdCheck(this, 'sid', Blockly.DBSP_SERVO_ID_DEFAULT, true)) {
			this.setWarningText(null);
		}
	}
};

Blockly.Blocks.dbsp_stop = {
	init: function () {
		this.tag = 'dbsp_stop';
		this.setColour(Blockly.Blocks.dbsp.HUE);
		this.appendValueInput('sid', Number)
			.appendField(Blockly.DBSP_STOP)
			.setCheck(Number);
		this.setPreviousStatement(true, null);
		this.setNextStatement(true, null);
		this.setTooltip(Blockly.DBSP_TOOLTIP_STOP);
	},

	onchange: function (changeEvent) {
		if (Blockly.Blocks.dbsp.initCheck(this) &&
			Blockly.Blocks.dbsp.servoIdCheck(this, 'sid', Blockly.DBSP_SERVO_ID_DEFAULT, true)) {
			this.setWarningText(null);
		}
	}
};