/*******************************************************************************
* Copyright (c) 2020 Mingyao Chuang
*
* All rights reserved.
* This program is free to use, but the ban on selling behavior.
* Modify the program must keep all the original text description,
* and can only comment out the original code (not allowed to delete).
*
* e-mail:mingyaochuang@gmail.com
*******************************************************************************/
#include "DBSPController.h"

#define BUFFER_SIZE							256
#define PACKET_HEADER						0x1d054c13

#define TRANSMISSION_MODE_SETTING_REQUEST	10010
#define TRANSMISSION_MODE_SETTING_RESPONSE	10011
#define QUERY_SERVO_LIST_REQUEST			20010
#define QUERY_SERVO_LIST_RESPONSE			20011
#define QUERY_SERVO_REQUEST					20020
#define QUERY_SERVO_RESPONSE				20021
#define SET_SERVO_PARAMETER_REQUEST			20030
#define SET_ANGLE_UNIT_REQUEST				20040
#define SET_ANGLE_UNIT_RESPONSE				20041
#define EXECUTE_ACTION_REQUEST				20050
#define EXECUTE_ACTIONEX_REQUEST			20060
#define EXECUTE_MACRO_REQUEST				30090
#define EXECUTE_MACRO_RESPONSE				30091
#define SUBSCRIBE_BUTTON_EVENT_REQUEST		40070
#define BUTTON_EVENT_RESPONSE				40081

#define CALLBACK(c, ...) if (c != NULL) { c(__VA_ARGS__); c = NULL; }

void DBSPController::begin(unsigned int rxPin, unsigned int txPin, unsigned long baud)
{
#ifdef SOFTWARE_SERIAL
	init();
	_serial = new SoftwareSerial(rxPin, txPin);
	_serial->begin(baud);
#endif
}

void DBSPController::begin(HardwareSerial * serial, unsigned long baud)
{
	init();
	_serial = serial;
	_serial->begin(baud);
}

void DBSPController::update()
{
	while (_serial->available() != 0)
	{
		handleByteFromServo(_serial->read());
	}
}

void DBSPController::transmissionModeSetting(byte mode, void(*callback)(byte))
{
	_modeSettingCallback = callback;
	makeHeader(TRANSMISSION_MODE_SETTING_REQUEST, 1);
	_txBuffer.write(mode);
	writeSerialData();
}

void DBSPController::subscribeButtonEvent(void(*callback)(long, byte))
{
	_buttonEventCallback = callback;
	makeHeader(SUBSCRIBE_BUTTON_EVENT_REQUEST, 1);
	_txBuffer.write(1);
	writeSerialData();
}

void DBSPController::queryServoList(void(*callback)(byte, const byte[]))
{
	_queryServoListCallback = callback;
	makeHeader(QUERY_SERVO_LIST_REQUEST, 0);
	writeSerialData();
}

void DBSPController::queryServo(byte id, void(*callback)(byte, const ServoInfo[]))
{
	_queryServoCallback = callback;
	makeHeader(QUERY_SERVO_REQUEST, 1);
	_txBuffer.write(id);
	writeSerialData();
}

void DBSPController::setAngleUnit(byte id, byte unitType, void(*callback)(byte, byte))
{
	_setAngleUnitCallback = callback;
	makeHeader(SET_ANGLE_UNIT_REQUEST, 2);
	_txBuffer.write(id);
	_txBuffer.write(unitType);
	writeSerialData();
}

void DBSPController::executeMacro(long id, void(*callback)(byte))
{
	_executeMacroCallback = callback;
	makeHeader(EXECUTE_MACRO_REQUEST, 4);
	_txBuffer.writeLong(id);
	writeSerialData();
}

void DBSPController::rotate(byte id, int degree, int interval)
{
	ActionContent arr[1];
	arr[0].id = id;
	arr[0].degree = degree;
	arr[0].interval = interval;
	executeAction(arr, 1);
}

void DBSPController::rotateBySpeed(byte id, int degree, int speed)
{
	ID2ARRAY(_isRotateBySpeed, id) = true;
	ID2ARRAY(_targetAngle, id) = degree;
	ID2ARRAY(_targetSpeed, id) = speed;
	readAngle(id, NULL);
}

void DBSPController::executeAction(long id)
{
	makeHeader(EXECUTE_ACTIONEX_REQUEST, 4);
	_txBuffer.writeLong(id);
	writeSerialData();
}

void DBSPController::executeAction(const ActionContent servos[], size_t servoCount)
{
	size_t contentSize = sizeof(ActionContent) * servoCount;
	makeHeader(EXECUTE_ACTION_REQUEST, sizeof(byte) + contentSize);
	_txBuffer.write(servoCount);
	_txBuffer.write(servos, contentSize);
	writeSerialData();
}

void DBSPController::readAngle(byte id, void(*callback)(byte, int))
{
	if (id != ALL_SERVOS)
	{
		_readAngleCallback = callback;
		makeHeader(QUERY_SERVO_REQUEST, 1);
		_txBuffer.write(id);
		writeSerialData();
	}
}

int DBSPController::readAngle(byte id)
{
	_readAngleNoCallback = true;
	readAngle(id, NULL);
	delay(50);
	while (_serial->available() != 0)
	{
		if (!_readAngleNoCallback) break;
		handleByteFromServo(_serial->read());
	}
	return _angle;
}

void DBSPController::release(byte id)
{
	setServoParameter(id, 0, 0);
}

void DBSPController::stop(byte id)
{
	setServoParameter(id, 0, 1);
}

void DBSPController::init()
{
	_txBuffer.init(BUFFER_SIZE);
	_rxBuffer.init(BUFFER_SIZE);
	_modeSettingCallback = NULL;
	_buttonEventCallback = NULL;
	_executeMacroCallback = NULL;
	_queryServoListCallback = NULL;
	_queryServoCallback = NULL;
	_setAngleUnitCallback = NULL;
	_readAngleCallback = NULL;
}

void DBSPController::handleByteFromServo(byte data)
{
	_rxBuffer.write(data);

	while (_rxBuffer.getLength() >= 9) // 9 for header & checksum.
	{
		if (_rxBuffer.readULong() == PACKET_HEADER)
		{
			unsigned int packetNumber = _rxBuffer.readUInt();
			unsigned int packetLength = _rxBuffer.readUInt();

			_rxBuffer.back(8);

			if (_rxBuffer.getLength() >= packetLength + 9) // + 9 for header & checksum.
			{
				// Do checksum.
				if (_rxBuffer.checksum())
				{
					// Pass header.
					_rxBuffer.forward(8);

					switch (packetNumber)
					{
					case TRANSMISSION_MODE_SETTING_RESPONSE:
					{
						byte result = _rxBuffer.read();
						CALLBACK(_modeSettingCallback, result);
						break;
					}
					case QUERY_SERVO_LIST_RESPONSE:
					{
						byte count = _rxBuffer.read();
						if (count <= 0)
						{
							CALLBACK(_queryServoListCallback, count, NULL);
						}
						else
						{
							byte *servos = new byte[count];
							_rxBuffer.read(servos, count);
							CALLBACK(_queryServoListCallback, count, servos);
							delete[] servos;
						}
						break;
					}
					case QUERY_SERVO_RESPONSE:
					{
						byte count = _rxBuffer.read();
						if (count <= 0)
						{
							CALLBACK(_queryServoCallback, count, NULL);
							if (_readAngleNoCallback)
							{
								_angle = 0;
								_readAngleNoCallback = false;
							}
							else
							{
								CALLBACK(_readAngleCallback, 0, 0);
							}
						}
						else
						{
							ServoInfo *servos = new ServoInfo[count];
							_rxBuffer.read(servos, sizeof(ServoInfo) * count);
							CALLBACK(_queryServoCallback, count, servos);
							if (_readAngleNoCallback)
							{
								_angle = servos[0].angle;
								_readAngleNoCallback = false;
							}
							else
							{
								CALLBACK(_readAngleCallback, servos[0].id, servos[0].angle);
							}
							if (ID2ARRAY(_isRotateBySpeed, servos[0].id) == true)
							{								
								ID2ARRAY(_isRotateBySpeed, servos[0].id) = false;
								rotateAfterReadAngle(servos[0].id, servos[0].angle);
							}
							delete[] servos;
						}
						break;
					}
					case SET_ANGLE_UNIT_RESPONSE:
					{
						byte id = _rxBuffer.read();
						byte unit = _rxBuffer.read();
						CALLBACK(_setAngleUnitCallback, id, unit);
						break;
					}
					case EXECUTE_MACRO_RESPONSE:
					{
						byte result = _rxBuffer.read();
						CALLBACK(_executeMacroCallback, result);
						break;
					}
					case BUTTON_EVENT_RESPONSE:
					{
						long button = _rxBuffer.readLong();
						byte event = _rxBuffer.read();
						CALLBACK(_buttonEventCallback, button, event);
						break;
					}
					default:
					{
						// Destroy all content.
						_rxBuffer.forward(packetLength);
						break;
					}
					}

					// Pass tail.
					_rxBuffer.forward(1);
				}
				else
				{
					// As checksum error occurs, destroy all content.
					_rxBuffer.forward(packetLength + 9);
				}
			}
			else
			{
				break;
			}
		}
		else
		{
			// Shift 1 byte(4 bytes for header).
			_rxBuffer.back(3);
		}
	}
}

void DBSPController::makeHeader(unsigned int number, unsigned int size)
{
	_txBuffer.clear();
	_txBuffer.writeULong(PACKET_HEADER);
	_txBuffer.writeUInt(number);
	_txBuffer.writeUInt(size);
}

void DBSPController::rotateAfterReadAngle(byte id, int angle)
{
	int diff = abs(ID2ARRAY(_targetAngle, id) - angle);
	double interval = ((double)diff / (double)ID2ARRAY(_targetSpeed, id)) * 100.0;
	rotate(id, ID2ARRAY(_targetAngle, id), (int)interval);
}

void DBSPController::setServoParameter(byte id, byte p1, byte p2)
{
	makeHeader(SET_SERVO_PARAMETER_REQUEST, 3);
	_txBuffer.write(id);
	_txBuffer.write(p1);
	_txBuffer.write(p2);
	writeSerialData();
}

void DBSPController::writeSerialData()
{
	_txBuffer.writeChecksum();
	while (_txBuffer.getLength() > 0)
	{
		byte c = _txBuffer.read();
		_serial->write(c);
	}
}
