/*******************************************************************************
* Copyright (c) 2020 Mingyao Chuang
*
* All rights reserved.
* This program is free to use, but the ban on selling behavior.
* Modify the program must keep all the original text description,
* and can only comment out the original code (not allowed to delete).
*
* e-mail:mingyaochuang@gmail.com
*******************************************************************************/
// DBSPController.h

#ifndef DBSPCONTROLLER_H
#define DBSPCONTROLLER_H

// If you want to use software serial port, uncomment the following line.
//#define SOFTWARE_SERIAL

#if defined(ARDUINO) && ARDUINO >= 100
#include "arduino.h"
#else
#include "WProgram.h"
#endif

#include "ByteBuffer.h"

#ifdef SOFTWARE_SERIAL
#include "SoftwareSerial.h"
#endif

/// Default connecting rate.
#define BAUD_RATE			57600

/// All servos.
#define ALL_SERVOS			0xff

/// Angle unit: degree.
#define ANGLE_UNIT_DEGREE	0
/// Angle unit: ADC.
#define ANGLE_UNIT_ADC		1

#define SERVO_STREAM_MAX	6
#define SERVO_ORDER_MAX		6

#define PSB_TRIANGLE		0x0001L
#define PSB_CIRCLE			0x0002L
#define PSB_CROSS			0x0004L
#define PSB_SQUARE			0x0008L
#define PSB_L1				0x0010L
#define PSB_R1				0x0020L
#define PSB_L2				0x0040L
#define PSB_R2				0x0080L
#define PSB_SELECT			0x0100L
#define PSB_START			0x0200L
#define PSB_L3				0x0400L
#define PSB_R3				0x0800L
#define PSB_PAD_UP			0x1000L
#define PSB_PAD_RIGHT		0x2000L
#define PSB_PAD_DOWN		0x4000L
#define PSB_PAD_LEFT		0x8000L

#define BUTTON_PRESS		1
#define BUTTON_LONG_PRESS	2
#define BUTTON_UP			4

#define SERVO_ID(stream, order) stream << 4 | order
#define ID2STREAM(id) id >> 4
#define ID2ORDER(id) id & 0xf
#define ID2ARRAY(arr, id) arr[ID2STREAM(id)][ID2ORDER(id)]

struct ActionContent
{
	byte id;
	int degree;
	int interval;
};

struct ServoInfo
{
	byte id;
	int angle;
	int current;
	int temperature;
	int servoType;
	byte isStart;
	byte isHold;
};

/*!
 * 
 * 
 */
class DBSPController
{
public:
	/*!
	 * Initializes the DBSP controller library and communication settings.
	 * It should be placed in function setup().
	 * This function is only enable as SOFTWARE_SERIAL be defined.
	 *
	 * \param rxPin Receive pin number.
	 * \param txPin Transmit pin number.
	 * \param baud Transmission rate, default value is BAUD_RATE.
	 */
	void begin(unsigned int rxPin, unsigned int txPin, unsigned long baud = BAUD_RATE);

	/*!
	 * Initializes the DBSP controller library and communication settings.
	 * It should be placed in function setup().
	 *
	 * \param serial A pointer to the class HardwareSerial, default value is Serial.
	 * \param baud Transmission rate, default value is BAUD_RATE.
	 */
	// void begin(HardwareSerial* serial = &Serial, unsigned long baud = BAUD_RATE);
	void begin(HardwareSerial * serial, unsigned long baud = BAUD_RATE);

	/*!
	 * Update data of this class.
	 * It should be placed in function loop().
	 */
	void update();

	/*!
	 * Set the controller to old or new transmission mode.
	 * Because the controller transmission mode is already set to the new mode, 
	 * this function will be called unless commands using the old mode is required.
	 *
	 * \param mode
	 * \param callback
	 */
	void transmissionModeSetting(byte mode, void(*callback)(byte));

	/*!
	 *
	 *
	 * \param callback
	 */
	void subscribeButtonEvent(void(*callback)(long, byte));

	/*!
	 *
	 *
	 * \param callback
	 */
	void queryServoList(void(*callback)(byte, const byte[]));

	/*!
	 *
	 *
	 * \param id
	 * \param callback
	 */
	void queryServo(byte id, void(*callback)(byte, const ServoInfo[]));

	/*!
	 *
	 *
	 * \param id
	 * \param unitType
	 * \param callback
	 */
	void setAngleUnit(byte id, byte unitType, void(*callback)(byte, byte));

	/*!
	 *
	 *
	 * \param id
	 * \param callback
	 */
	void executeMacro(long id, void(*callback)(byte) = NULL);

	/*!
	 *
	 *
	 * \param id
	 * \param degree
	 * \param interval
	 */
	void rotate(byte id, int degree, int interval);

	/*!
	 * 
	 * 
	 * \param id
	 * \param degree
	 * \param speed
	 */
	void rotateBySpeed(byte id, int degree, int speed);

	/*!
	 *
	 *
	 * \param id
	 */
	void executeAction(long id);

	/*!
	 *
	 *
	 * \param servos
	 * \param servoCount
	 */
	void executeAction(const ActionContent servos[], size_t servoCount);

	/*!
	 *
	 *
	 * \param id
	 * \param callback
	 */
	void readAngle(byte id, void(*callback)(byte, int));

	/*!
	 * 
	 * 
	 * \param id
	 * \return 
	 */
	int readAngle(byte id);

	/*!
	 *
	 *
	 * \param id
	 */
	void release(byte id = ALL_SERVOS);

	/*!
	 * 
	 * 
	 * \param id
	 */
	void stop(byte id = ALL_SERVOS);

private:

#ifdef SOFTWARE_SERIAL
	SoftwareSerial* _serial;
#else
	HardwareSerial* _serial;
#endif

	ByteBuffer _txBuffer;
	ByteBuffer _rxBuffer;

	int _angle;
	bool _readAngleNoCallback;
	int _targetAngle[SERVO_STREAM_MAX][SERVO_ORDER_MAX];
	int _targetSpeed[SERVO_STREAM_MAX][SERVO_ORDER_MAX];
	bool _isRotateBySpeed[SERVO_STREAM_MAX][SERVO_ORDER_MAX];

	void(*_modeSettingCallback)(byte);
	void(*_buttonEventCallback)(long, byte);
	void(*_executeMacroCallback)(byte);
	void(*_queryServoListCallback)(byte, const byte[]);
	void(*_queryServoCallback)(byte, const ServoInfo[]);
	void(*_setAngleUnitCallback)(byte, byte);
	void(*_readAngleCallback)(byte, int);
	
	void init();
	void handleByteFromServo(byte data);
	void makeHeader(unsigned int number, unsigned int size);
	void rotateAfterReadAngle(byte id, int angle);
	void setServoParameter(byte id, byte p1, byte p2);
	void writeSerialData();
};

#endif